// Create context menu on install
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: "dwa-report-bug",
    title: "Report Bug to DWA Builder",
    contexts: ["page", "image", "selection", "link"]
  });

  chrome.contextMenus.create({
    id: "dwa-record-video",
    title: "Record Video for DWA Builder",
    contexts: ["page", "image", "selection", "link"]
  });
});

// Handle context menu click
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId === "dwa-record-video") {
    // Ensure recorder.js is injected (handles tabs opened before extension was installed)
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        files: ["recorder.js"]
      });
    } catch (e) {
      console.log("Script injection note:", e.message);
    }
    // Small delay to let script initialize
    setTimeout(() => {
      chrome.tabs.sendMessage(tab.id, {
        action: "startRecording",
        pageUrl: tab.url,
        pageTitle: tab.title
      });
    }, 100);
    return;
  }

  if (info.menuItemId !== "dwa-report-bug") return;

  try {
    // Inject content script for region selection
    await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      files: ["content.js"]
    }).catch(() => {});

    const screenshotDataUrl = await chrome.tabs.captureVisibleTab(tab.windowId, {
      format: "png"
    });

    // Store screenshot in storage (avoids message size limits)
    await chrome.storage.local.set({
      capturedScreenshot: screenshotDataUrl
    });

    // Tell content script to start region selection (small message, no screenshot)
    chrome.tabs.sendMessage(tab.id, {
      action: "startRegionSelect",
      pageUrl: tab.url,
      pageTitle: tab.title
    });
  } catch (err) {
    console.error("Failed to capture screenshot:", err);
  }
});

// Listen for messages from content script
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === "openReportWindow") {
    const popupUrl = chrome.runtime.getURL("popup.html");
    chrome.windows.create({
      url: popupUrl,
      type: "popup",
      width: 400,
      height: 560,
      focused: true
    });
    sendResponse({ ok: true });
    return false;
  }

  if (message.action === "regionCancelled") {
    sendResponse({ ok: true });
    return false;
  }
});
