(() => {
  let overlay = null;
  let selectionBox = null;
  let hintText = null;
  let isSelecting = false;
  let startX = 0;
  let startY = 0;
  let pageUrl = "";
  let pageTitle = "";

  // Listen for messages from background script
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.action === "startRegionSelect") {
      pageUrl = message.pageUrl || "";
      pageTitle = message.pageTitle || "";
      createOverlay();
      sendResponse({ ok: true });
      return false;
    }
  });

  function createOverlay() {
    removeOverlay();

    overlay = document.createElement("div");
    overlay.id = "dwa-bug-reporter-overlay";
    Object.assign(overlay.style, {
      position: "fixed",
      top: "0",
      left: "0",
      width: "100vw",
      height: "100vh",
      backgroundColor: "rgba(0, 0, 0, 0.4)",
      zIndex: "2147483647",
      cursor: "crosshair",
      userSelect: "none"
    });

    hintText = document.createElement("div");
    Object.assign(hintText.style, {
      position: "fixed",
      top: "50%",
      left: "50%",
      transform: "translate(-50%, -50%)",
      color: "#ffffff",
      fontSize: "16px",
      fontFamily: "-apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif",
      fontWeight: "500",
      padding: "12px 24px",
      backgroundColor: "rgba(26, 26, 46, 0.9)",
      borderRadius: "8px",
      border: "1px solid rgba(74, 158, 255, 0.3)",
      pointerEvents: "none",
      zIndex: "2147483647",
      textAlign: "center",
      lineHeight: "1.5",
      backdropFilter: "blur(4px)"
    });
    hintText.textContent = "Drag to select a region \u2022 Press Escape to cancel";

    selectionBox = document.createElement("div");
    Object.assign(selectionBox.style, {
      position: "fixed",
      border: "2px solid #4a9eff",
      backgroundColor: "rgba(74, 158, 255, 0.15)",
      zIndex: "2147483647",
      pointerEvents: "none",
      display: "none",
      borderRadius: "2px",
      boxShadow: "0 0 0 1px rgba(74, 158, 255, 0.3)"
    });

    document.body.appendChild(overlay);
    document.body.appendChild(hintText);
    document.body.appendChild(selectionBox);

    overlay.addEventListener("mousedown", onMouseDown);
    overlay.addEventListener("mousemove", onMouseMove);
    overlay.addEventListener("mouseup", onMouseUp);
    document.addEventListener("keydown", onKeyDown);
  }

  function onMouseDown(e) {
    e.preventDefault();
    isSelecting = true;
    startX = e.clientX;
    startY = e.clientY;

    if (hintText) {
      hintText.style.display = "none";
    }

    selectionBox.style.display = "block";
    selectionBox.style.left = startX + "px";
    selectionBox.style.top = startY + "px";
    selectionBox.style.width = "0px";
    selectionBox.style.height = "0px";
  }

  function onMouseMove(e) {
    if (!isSelecting) return;
    e.preventDefault();

    const left = Math.min(startX, e.clientX);
    const top = Math.min(startY, e.clientY);
    const width = Math.abs(e.clientX - startX);
    const height = Math.abs(e.clientY - startY);

    selectionBox.style.left = left + "px";
    selectionBox.style.top = top + "px";
    selectionBox.style.width = width + "px";
    selectionBox.style.height = height + "px";
  }

  function onMouseUp(e) {
    if (!isSelecting) return;
    isSelecting = false;

    const left = Math.min(startX, e.clientX);
    const top = Math.min(startY, e.clientY);
    const width = Math.abs(e.clientX - startX);
    const height = Math.abs(e.clientY - startY);

    if (width < 10 || height < 10) {
      removeOverlay();
      return;
    }

    // Read the screenshot from storage, crop it, then open report window
    removeOverlay();
    cropAndSubmit(left, top, width, height);
  }

  function onKeyDown(e) {
    if (e.key === "Escape") {
      removeOverlay();
    }
  }

  function cropAndSubmit(x, y, w, h) {
    chrome.storage.local.get("capturedScreenshot", (data) => {
      if (!data.capturedScreenshot) {
        console.error("No screenshot found in storage");
        return;
      }

      const dpr = window.devicePixelRatio || 1;
      const img = new Image();

      img.onload = () => {
        const canvas = document.createElement("canvas");
        canvas.width = w * dpr;
        canvas.height = h * dpr;

        const ctx = canvas.getContext("2d");
        ctx.drawImage(
          img,
          x * dpr, y * dpr,
          w * dpr, h * dpr,
          0, 0,
          w * dpr, h * dpr
        );

        const croppedDataUrl = canvas.toDataURL("image/png");

        // Store cropped screenshot for the popup, then open report window
        chrome.storage.local.set({
          pendingReport: {
            screenshot: croppedDataUrl,
            pageUrl: pageUrl,
            pageTitle: pageTitle,
            timestamp: Date.now()
          }
        }, () => {
          // Clean up the full screenshot
          chrome.storage.local.remove("capturedScreenshot");
          // Tell background to open the report window
          chrome.runtime.sendMessage({ action: "openReportWindow" });
        });
      };

      img.onerror = () => {
        console.error("Failed to load screenshot for cropping");
      };

      img.src = data.capturedScreenshot;
    });
  }

  function removeOverlay() {
    if (overlay) {
      overlay.removeEventListener("mousedown", onMouseDown);
      overlay.removeEventListener("mousemove", onMouseMove);
      overlay.removeEventListener("mouseup", onMouseUp);
      overlay.remove();
      overlay = null;
    }
    if (selectionBox) {
      selectionBox.remove();
      selectionBox = null;
    }
    if (hintText) {
      hintText.remove();
      hintText = null;
    }
    document.removeEventListener("keydown", onKeyDown);
  }
})();
