document.addEventListener("DOMContentLoaded", async () => {
  const form = document.getElementById("report-form");
  const noReport = document.getElementById("no-report");
  const screenshotImg = document.getElementById("screenshot-img");
  const projectSelect = document.getElementById("project-select");
  const typeSelect = document.getElementById("type-select");
  const description = document.getElementById("description");
  const pageUrlInput = document.getElementById("page-url");
  const submitBtn = document.getElementById("submit-btn");
  const cancelBtn = document.getElementById("cancel-btn");
  const statusMessage = document.getElementById("status-message");
  const openSettings = document.getElementById("open-settings");

  let settings = {};
  let pendingReport = null;
  let isVideoMode = false;

  // Load settings
  const syncData = await chrome.storage.sync.get(["dwaUrl", "dwaApiKey", "reporterName", "reporterEmail"]);
  settings.dwaUrl = syncData.dwaUrl || "";
  settings.dwaApiKey = syncData.dwaApiKey || "";
  settings.reporterName = syncData.reporterName || "";
  settings.reporterEmail = syncData.reporterEmail || "";

  // Load pending report
  const localData = await chrome.storage.local.get(["pendingReport"]);
  pendingReport = localData.pendingReport || null;

  if (pendingReport && (pendingReport.screenshot || (pendingReport.type === "video" && pendingReport.videoUrl))) {
    // Show the form
    form.style.display = "block";
    noReport.style.display = "none";

    if (pendingReport.type === "video" && pendingReport.videoUrl) {
      // Video mode — video already uploaded, we just have the URL
      document.querySelector(".screenshot-preview").style.display = "none";
      document.querySelector(".video-preview").style.display = "block";
      // If URL is relative, prepend the DWA Builder base URL
      let videoSrc = pendingReport.videoUrl;
      if (videoSrc.startsWith("/") && settings.dwaUrl) {
        videoSrc = settings.dwaUrl.replace(/\/$/, "") + videoSrc;
      }
      document.getElementById("video-preview").src = videoSrc;
      isVideoMode = true;
    } else {
      screenshotImg.src = pendingReport.screenshot;
    }

    pageUrlInput.value = pendingReport.pageUrl || "";

    // Clear the badge
    chrome.action.setBadgeText({ text: "" });

    // Fetch apps from API, auto-select based on page URL, then fetch types
    await fetchApps(pendingReport.pageUrl);
    fetchTypes(projectSelect.value);
  } else {
    form.style.display = "none";
    noReport.style.display = "block";
  }

  // App change -> refetch types
  projectSelect.addEventListener("change", () => {
    fetchTypes(projectSelect.value);
  });

  async function fetchApps(pageUrl) {
    if (!settings.dwaUrl) return;

    try {
      const url = `${settings.dwaUrl.replace(/\/$/, "")}/api/repos`;
      const headers = {};
      if (settings.dwaApiKey) {
        headers["Authorization"] = `Bearer ${settings.dwaApiKey}`;
      }

      const resp = await fetch(url, { headers });
      if (resp.ok) {
        const repos = await resp.json();
        projectSelect.innerHTML = "";

        if (!Array.isArray(repos) || repos.length === 0) {
          projectSelect.innerHTML = '<option value="">No apps configured</option>';
          return;
        }

        // Find which app matches the page URL (longest pattern first so subdomains beat bare domains)
        let matchedApp = null;
        if (pageUrl) {
          const sorted = [...repos].sort((a, b) => (b.url_pattern || "").length - (a.url_pattern || "").length);
          for (const r of sorted) {
            const pattern = r.url_pattern || "";
            if (pattern && pageUrl.includes(pattern)) {
              matchedApp = r.name;
              break;
            }
          }
        }

        // Build options — matched app first, then others, then Uncategorized at the end
        repos.forEach(r => {
          const opt = document.createElement("option");
          opt.value = r.name || r;
          opt.textContent = r.name || r;
          projectSelect.appendChild(opt);
        });

        // Add "Uncategorized" as fallback option
        const uncatOpt = document.createElement("option");
        uncatOpt.value = "Uncategorized";
        uncatOpt.textContent = "Uncategorized";
        projectSelect.appendChild(uncatOpt);

        // Auto-select matched app, or "Uncategorized" if no match
        if (matchedApp) {
          projectSelect.value = matchedApp;
        } else {
          projectSelect.value = "Uncategorized";
        }
      }
    } catch (e) {
      console.error("Failed to fetch apps:", e);
    }
  }

  async function fetchTypes(repo) {
    if (!settings.dwaUrl) return;

    try {
      const url = `${settings.dwaUrl.replace(/\/$/, "")}/api/types/${encodeURIComponent(repo)}`;
      const headers = {};
      if (settings.dwaApiKey) {
        headers["Authorization"] = `Bearer ${settings.dwaApiKey}`;
      }

      const resp = await fetch(url, { headers });
      if (resp.ok) {
        const types = await resp.json();
        typeSelect.innerHTML = "";
        if (Array.isArray(types) && types.length > 0) {
          types.forEach(t => {
            const opt = document.createElement("option");
            opt.value = t.name || t;
            opt.textContent = t.display_name || t.name || t;
            typeSelect.appendChild(opt);
          });
          // Default to "bug" if available
          const bugOpt = typeSelect.querySelector('option[value="bug"]');
          if (bugOpt) bugOpt.selected = true;
        } else {
          addDefaultTypes();
        }
      } else {
        addDefaultTypes();
      }
    } catch (e) {
      addDefaultTypes();
    }
  }

  function addDefaultTypes() {
    typeSelect.innerHTML = "";
    ["bug", "feature", "task", "improvement"].forEach(t => {
      const opt = document.createElement("option");
      opt.value = t;
      opt.textContent = t.charAt(0).toUpperCase() + t.slice(1);
      typeSelect.appendChild(opt);
    });
  }

  // Submit
  form.addEventListener("submit", async (e) => {
    e.preventDefault();

    if (!settings.dwaUrl) {
      showStatus("error", "Please configure the DWA Builder URL in extension settings.");
      return;
    }

    const desc = description.value.trim();

    submitBtn.disabled = true;

    const subject = desc ? desc.split("\n")[0].substring(0, 200) : "";
    const fullDescription = (desc || "") + "\n\nPage: " + (pendingReport.pageUrl || "N/A");

    let payload;

    if (isVideoMode) {
      // Video already uploaded by recorder.js — just submit with the URL
      showStatus("loading", "Submitting report...");
      payload = {
        type: typeSelect.value,
        subject: subject,
        description: fullDescription,
        repo: projectSelect.value,
        video_url: pendingReport.videoUrl,
        reporter_name: settings.reporterName || undefined,
        reporter_email: settings.reporterEmail || undefined
      };
    } else {
      showStatus("loading", "Submitting report...");
      payload = {
        type: typeSelect.value,
        subject: subject,
        description: fullDescription,
        repo: projectSelect.value,
        screenshot: pendingReport.screenshot,
        reporter_name: settings.reporterName || undefined,
        reporter_email: settings.reporterEmail || undefined
      };
    }

    try {
      const url = `${settings.dwaUrl.replace(/\/$/, "")}/api/feedback`;
      const headers = {
        "Content-Type": "application/json"
      };
      if (settings.dwaApiKey) {
        headers["Authorization"] = `Bearer ${settings.dwaApiKey}`;
      }

      const resp = await fetch(url, {
        method: "POST",
        headers,
        body: JSON.stringify(payload)
      });

      if (resp.ok) {
        showStatus("success", "Bug report submitted successfully!");
        // Clear pending report
        await chrome.storage.local.remove(["pendingReport"]);
        // Close popup after a short delay
        setTimeout(() => window.close(), 1500);
      } else {
        const text = await resp.text();
        showStatus("error", `Failed (${resp.status}): ${text.substring(0, 100)}`);
        submitBtn.disabled = false;
      }
    } catch (err) {
      showStatus("error", `Network error: ${err.message}`);
      submitBtn.disabled = false;
    }
  });

  // Cancel
  cancelBtn.addEventListener("click", async () => {
    await chrome.storage.local.remove(["pendingReport"]);
    chrome.action.setBadgeText({ text: "" });
    window.close();
  });

  // Open settings
  openSettings.addEventListener("click", (e) => {
    e.preventDefault();
    chrome.runtime.openOptionsPage();
  });

  function showStatus(type, message) {
    statusMessage.className = "status-message " + type;
    statusMessage.textContent = message;
  }
});
