(function() {
  if (window.__dwaRecorderLoaded) return;
  window.__dwaRecorderLoaded = true;

  let mediaRecorder = null;
  let recordedChunks = [];
  let controlBar = null;
  let timerInterval = null;
  let startTime = 0;

  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.action === "startRecording") {
      startRecording(message.pageUrl, message.pageTitle);
      sendResponse({ ok: true });
    }
    return false;
  });

  async function startRecording(pageUrl, pageTitle) {
    try {
      const stream = await navigator.mediaDevices.getDisplayMedia({
        video: { cursor: "always" },
        audio: true
      });

      // Also capture microphone
      try {
        const micStream = await navigator.mediaDevices.getUserMedia({
          audio: {
            echoCancellation: true,
            noiseSuppression: true,
            autoGainControl: true,
            sampleRate: 48000
          }
        });

        // Mix system audio + mic using AudioContext with gain control
        const audioContext = new AudioContext({ sampleRate: 48000 });
        const dest = audioContext.createMediaStreamDestination();

        // System audio at normal volume
        stream.getAudioTracks().forEach(track => {
          const source = audioContext.createMediaStreamSource(new MediaStream([track]));
          const gain = audioContext.createGain();
          gain.gain.value = 1.0;
          source.connect(gain);
          gain.connect(dest);
        });

        // Mic audio with gain control
        const micSource = audioContext.createMediaStreamSource(micStream);
        const micGain = audioContext.createGain();
        micGain.gain.value = 1.0;
        micSource.connect(micGain);
        micGain.connect(dest);

        const combinedStream = new MediaStream([
          ...stream.getVideoTracks(),
          ...dest.stream.getAudioTracks()
        ]);

        startMediaRecorder(combinedStream, pageUrl, pageTitle, () => {
          micStream.getTracks().forEach(t => t.stop());
          stream.getTracks().forEach(t => t.stop());
          audioContext.close();
        });
      } catch (micErr) {
        // No mic permission — just record with display audio (or none)
        startMediaRecorder(stream, pageUrl, pageTitle, () => {
          stream.getTracks().forEach(t => t.stop());
        });
      }
    } catch (err) {
      console.error("Screen capture cancelled or failed:", err);
    }
  }

  function startMediaRecorder(stream, pageUrl, pageTitle, cleanupFn) {
    recordedChunks = [];

    mediaRecorder = new MediaRecorder(stream, {
      mimeType: "video/webm;codecs=vp9,opus",
      audioBitsPerSecond: 128000,
      videoBitsPerSecond: 5000000
    });

    mediaRecorder.ondataavailable = (e) => {
      if (e.data.size > 0) recordedChunks.push(e.data);
    };

    mediaRecorder.onstop = async () => {
      cleanupFn();
      removeControlBar();

      const blob = new Blob(recordedChunks, { type: "video/webm" });
      recordedChunks = [];

      // Show uploading indicator
      showUploadingBar();

      try {
        // Get DWA URL from settings
        const syncData = await chrome.storage.sync.get(["dwaUrl", "dwaApiKey"]);
        const dwaUrl = syncData.dwaUrl || "";
        if (!dwaUrl) {
          removeUploadingBar();
          alert("DWA Builder URL not configured. Go to extension options to set it.");
          return;
        }

        // Upload video directly to server
        const formData = new FormData();
        formData.append("file", blob, "recording.webm");

        const uploadHeaders = {};
        if (syncData.dwaApiKey) uploadHeaders["Authorization"] = `Bearer ${syncData.dwaApiKey}`;

        const resp = await fetch(`${dwaUrl.replace(/\/$/, "")}/api/upload-video`, {
          method: "POST",
          headers: uploadHeaders,
          body: formData
        });

        removeUploadingBar();

        if (!resp.ok) {
          const errText = await resp.text();
          alert("Video upload failed: " + errText);
          return;
        }

        const result = await resp.json();

        // Store just the URL and metadata (tiny, fits in storage)
        await chrome.storage.local.set({
          pendingReport: {
            type: "video",
            videoUrl: result.url,
            pageUrl: pageUrl,
            pageTitle: pageTitle
          }
        });

        // Open popup window
        chrome.runtime.sendMessage({ action: "openReportWindow" });
      } catch (err) {
        removeUploadingBar();
        console.error("Video upload failed:", err);
        alert("Video upload failed: " + err.message);
      }
    };

    // Handle stream ending (user clicks browser's "Stop sharing")
    stream.getVideoTracks()[0].onended = () => {
      if (mediaRecorder && mediaRecorder.state !== "inactive") {
        mediaRecorder.stop();
      }
    };

    mediaRecorder.start(1000); // collect data every second
    startTime = Date.now();
    showControlBar();
  }

  function showControlBar() {
    controlBar = document.createElement("div");
    controlBar.id = "dwa-recorder-bar";
    controlBar.innerHTML = `
      <div style="display:flex;align-items:center;gap:10px;background:#1a1a2e;border:2px solid #f85149;border-radius:12px;padding:8px 16px;color:#fff;font-family:-apple-system,BlinkMacSystemFont,sans-serif;font-size:14px;box-shadow:0 4px 20px rgba(0,0,0,0.5);z-index:2147483647;position:fixed;bottom:20px;right:20px;">
        <div style="width:12px;height:12px;background:#f85149;border-radius:50%;animation:dwa-pulse 1s infinite;"></div>
        <span id="dwa-timer" style="font-variant-numeric:tabular-nums;min-width:48px;">0:00</span>
        <button id="dwa-stop-btn" style="background:#f85149;color:#fff;border:none;border-radius:6px;padding:6px 14px;font-size:13px;font-weight:600;cursor:pointer;">Stop</button>
      </div>
    `;

    const style = document.createElement("style");
    style.textContent = "@keyframes dwa-pulse{0%,100%{opacity:1}50%{opacity:0.3}}";
    controlBar.appendChild(style);

    document.body.appendChild(controlBar);

    document.getElementById("dwa-stop-btn").addEventListener("click", () => {
      if (mediaRecorder && mediaRecorder.state !== "inactive") {
        mediaRecorder.stop();
      }
    });

    timerInterval = setInterval(() => {
      const elapsed = Math.floor((Date.now() - startTime) / 1000);
      const mins = Math.floor(elapsed / 60);
      const secs = elapsed % 60;
      const el = document.getElementById("dwa-timer");
      if (el) el.textContent = `${mins}:${secs.toString().padStart(2, "0")}`;
    }, 1000);
  }

  let uploadingBar = null;

  function showUploadingBar() {
    uploadingBar = document.createElement("div");
    uploadingBar.innerHTML = `
      <div style="display:flex;align-items:center;gap:10px;background:#1a1a2e;border:2px solid #388bfd;border-radius:12px;padding:8px 16px;color:#fff;font-family:-apple-system,BlinkMacSystemFont,sans-serif;font-size:14px;box-shadow:0 4px 20px rgba(0,0,0,0.5);z-index:2147483647;position:fixed;bottom:20px;right:20px;">
        <div style="width:12px;height:12px;background:#388bfd;border-radius:50%;animation:dwa-pulse 1s infinite;"></div>
        <span>Uploading video...</span>
      </div>
    `;
    document.body.appendChild(uploadingBar);
  }

  function removeUploadingBar() {
    if (uploadingBar) {
      uploadingBar.remove();
      uploadingBar = null;
    }
  }

  function removeControlBar() {
    if (timerInterval) {
      clearInterval(timerInterval);
      timerInterval = null;
    }
    if (controlBar) {
      controlBar.remove();
      controlBar = null;
    }
  }
})();
